<?php

namespace Vote4Rewards\Pingback;

class CallbackVerifier
{
    /**
     * Allowed IP addresses from vote4rewards.de
     * Update this list if vote4rewards.de changes their callback servers
     */
    private array $allowedIps = [
        '91.98.130.153',
    ];

    /**
     * Set allowed IP addresses for callback verification
     *
     * @param array $ips Array of allowed IP addresses
     */
    public function setAllowedIps(array $ips): self
    {
        $this->allowedIps = $ips;
        return $this;
    }

    /**
     * Add an allowed IP address
     *
     * @param string $ip IP address to allow
     */
    public function addAllowedIp(string $ip): self
    {
        $this->allowedIps[] = $ip;
        return $this;
    }

    /**
     * Verify if the request comes from an allowed IP
     *
     * @param string $requestIp The IP address of the incoming request
     * @return bool True if IP is allowed
     */
    public function verifyIp(string $requestIp): bool
    {
        // If no IPs configured, skip IP verification
        if (empty($this->allowedIps)) {
            return true;
        }

        return in_array($requestIp, $this->allowedIps, true);
    }

    /**
     * Verify callback data integrity
     *
     * @param array $data Callback data
     * @return bool True if data is valid
     */
    public function verifyData(array $data): bool
    {
        // Check required fields
        $requiredFields = ['voter_id', 'server_uuid'];

        foreach ($requiredFields as $field) {
            if (!isset($data[$field]) || empty($data[$field])) {
                return false;
            }
        }

        return true;
    }

    /**
     * Verify signature if provided (optional security feature)
     * vote4rewards.de can sign callbacks with a secret key
     *
     * @param array $data Callback data
     * @param string $signature Provided signature
     * @param string $secret Your shared secret with vote4rewards.de
     * @return bool True if signature is valid
     */
    public function verifySignature(array $data, string $signature, string $secret): bool
    {
        // Create expected signature
        $payload = json_encode($data);
        $expectedSignature = hash_hmac('sha256', $payload, $secret);

        return hash_equals($expectedSignature, $signature);
    }

    /**
     * Complete verification (IP + Data)
     *
     * @param array $data Callback data
     * @param string $requestIp Request IP address
     * @return bool True if all verifications pass
     */
    public function verify(array $data, string $requestIp): bool
    {
        return $this->verifyIp($requestIp) && $this->verifyData($data);
    }
}
