<?php

namespace Vote4Rewards\Pingback;

use Closure;

class CallbackHandler
{
    private ?Closure $rewardCallback = null;
    private ?Closure $verificationCallback = null;

    /**
     * Set the callback function that will be called when a reward is received
     *
     * @param Closure $callback Function signature: function(array $data): bool
     */
    public function onReward(Closure $callback): self
    {
        $this->rewardCallback = $callback;
        return $this;
    }

    /**
     * Set a custom verification callback (optional)
     * If not set, basic IP verification will be used
     *
     * @param Closure $callback Function signature: function(array $data, string $ip): bool
     */
    public function onVerify(Closure $callback): self
    {
        $this->verificationCallback = $callback;
        return $this;
    }

    /**
     * Handle incoming callback from vote4rewards.de
     *
     * @param array $data The callback data (usually $_POST or request data)
     * @param string $requestIp The IP address of the request
     * @param ?string $webhookSecret The shared webhook secret for signature verification
     * @param ?string $signature The X-Webhook-Signature header value
     * @return array Response array with success status and message
     */
    public function handle(array $data, string $requestIp, ?string $webhookSecret = null, ?string $signature = null): array
    {
        if (!isset($data['voter_id']) || !isset($data['server_uuid'])) {
            return [
                'success' => false,
                'message' => 'Missing required fields: voter_id and server_uuid are required'
            ];
        }

        if ($webhookSecret && $signature) {
            if (!$this->verifySignature($data, $signature, $webhookSecret)) {
                return [
                    'success' => false,
                    'message' => 'Invalid webhook signature'
                ];
            }
        } elseif ($webhookSecret && !$signature) {
            return [
                'success' => false,
                'message' => 'Missing X-Webhook-Signature header'
            ];
        }

        if ($this->verificationCallback !== null) {
            $verified = call_user_func($this->verificationCallback, $data, $requestIp);
            if (!$verified) {
                return [
                    'success' => false,
                    'message' => 'Verification failed'
                ];
            }
        }

        if ($this->rewardCallback !== null) {
            try {
                $result = call_user_func($this->rewardCallback, $data);

                if ($result) {
                    return [
                        'success' => true,
                        'message' => 'Reward processed successfully'
                    ];
                } else {
                    return [
                        'success' => false,
                        'message' => 'Reward processing failed'
                    ];
                }
            } catch (\Exception $e) {
                return [
                    'success' => false,
                    'message' => 'Error processing reward: ' . $e->getMessage()
                ];
            }
        }

        return [
            'success' => false,
            'message' => 'No reward callback configured'
        ];
    }

    /**
     * Verify the HMAC webhook signature
     *
     * @param array $data The callback data
     * @param string $signature The signature header value (e.g., 'sha256=abc123...')
     * @param string $secret The shared webhook secret
     * @return bool True if signature is valid
     */
    private function verifySignature(array $data, string $signature, string $secret): bool
    {
        if (strpos($signature, '=') === false) {
            return false;
        }

        [$algorithm, $hash] = explode('=', $signature, 2);

        if ($algorithm !== 'sha256') {
            return false;
        }

        $payload = json_encode($data);
        $expectedHash = hash_hmac('sha256', $payload, $secret);

        return hash_equals($expectedHash, $hash);
    }

    /**
     * Get the standardized callback data structure
     *
     * @param array $rawData Raw callback data
     * @return array Normalized callback data
     */
    public static function normalizeCallbackData(array $rawData): array
    {
        return [
            'server_uuid' => $rawData['server_uuid'] ?? null,       // Your server UUID on vote4rewards.de
            'voter_id' => $rawData['voter_id'] ?? null,             // Unique identifier of the voter
            'ip_address' => $rawData['ip_address'] ?? null,         // Voter's IP address
            'voted_at' => $rawData['voted_at'] ?? null,             // Vote timestamp
        ];
    }
}
