# vote4rewards/pingback

A PHP package for integrating vote4rewards.de voting and reward callbacks into your game server website.

## Features

- Generate voting links with rewarder parameters using server UUID
- Handle reward callbacks from vote4rewards.de
- Built-in security verification (IP whitelist, signature verification)
- Simple and flexible API
- Framework-agnostic (works with Laravel, plain PHP, etc.)
- **Built-in testing tool** in the vote4rewards.de dashboard

## Installation

```bash
composer config repositories.vote4rewards composer https://composer.vote4rewards.de
composer require vote4rewards/pingback
```

## Quick Start

### 1. Get Your Server UUID

After registering your server on vote4rewards.de, navigate to your server dashboard and click on the **"Package"** button to access the integration page.

**What you'll find there:**

- Your unique Server UUID (automatically generated)
- Copy-paste ready code examples
- Installation instructions
- **Live testing tool** to verify your callback URL works correctly

### 2. Generate Vote Links

Allow your players to vote on vote4rewards.de with their in-game username:

```php
use Vote4Rewards\Pingback\LinkGenerator;

$linkGenerator = new LinkGenerator();

// Generate a vote link for a specific player
// Replace 'your-server-uuid' with your actual server UUID
$voteUrl = $linkGenerator->generateVoteLink('550e8400-e29b-41d4-a716-446655440000', 'PlayerUsername');
// Result: https://vote4rewards.de/vote/550e8400-e29b-41d4-a716-446655440000?rewarder=PlayerUsername

// Display the link to your players
echo '<a href="' . $voteUrl . '">Vote for us!</a>';
```

### 3. Receive Reward Callbacks

When a player votes, vote4rewards.de sends a callback to your server. Handle it like this:

```php
use Vote4Rewards\Pingback\CallbackHandler;

// Create callback handler
$handler = new CallbackHandler();
$verifier = new CallbackVerifier();

// Define the verification
$handler->onVerify(function (array $data, string $ip) use ($verifier) {
    return $verifier->verify($data, $ip);
});

// Define what happens when a reward is received
$handler->onReward(function(array $data) {
    // Extract player username and server info
    $username = $data['voter_id'];
    $serverId = $data['server_uuid'];
    
    // Give reward to player (example with database)
    $pdo = new PDO('mysql:host=localhost;dbname=yourdb', 'user', 'pass');
    $stmt = $pdo->prepare('UPDATE players SET vote_points = vote_points + 1 WHERE username = ?');
    $result = $stmt->execute([$username]);
    
    // Return true if reward was given successfully
    return $result;
});

// Get your webhook secret from vote4rewards.de dashboard
$webhookSecret = '<your_webhook_secret_from_vote4rewards_dashboard>';
$signature = $_SERVER['HTTP_X_WEBHOOK_SIGNATURE'] ?? null;

// Handle the incoming callback with signature verification
$result = $handler->handle(
    $_POST,
    $_SERVER['REMOTE_ADDR'],
    $webhookSecret,
    $signature
);

// Send response with correct HTTP status code
http_response_code($result['success'] ? 200 : 401);
header('Content-Type: application/json');
echo json_encode($result);
```

## Laravel Integration

### Step 1: Add Route

In `routes/web.php`:

```php
use Illuminate\Http\Request;
use Vote4Rewards\Pingback\CallbackHandler;

Route::post('/vote/callback', function (Request $request) {
    $handler = new CallbackHandler();
    $verifier = new CallbackVerifier();

    $handler->onVerify(function (array $data, string $ip) use ($verifier) {
        return $verifier->verify($data, $ip);
    });
    
    $handler->onReward(function(array $data) {
        // Give reward using Laravel database
        \DB::table('players')
            ->where('username', $data['voter_id'])
            ->increment('vote_points');
        
        return true;
    });
    
    // Get webhook secret from config and signature from headers
    $webhookSecret = config('vote4rewards.webhook_secret');
    $signature = $request->header('X-Webhook-Signature');
    
    $result = $handler->handle(
        $request->all(),
        $request->ip(),
        $webhookSecret,
        $signature
    );
    
    return response()->json($result, $result['success'] ? 200 : 401);
})->name('vote.callback');
```

### Step 2: Disable CSRF for Callback Route

In `bootstrap/app.php`:

```php
->withMiddleware(function (Middleware $middleware) {
    $middleware->validateCsrfTokens(except: [
        'vote/callback',
    ]);
})
```

Older Laravel Version:
In `app/Http/Middleware/VerifyCsrfToken.php`:

```php
protected $except = [
    'vote/callback',
];
```

### Step 3: Generate Vote Links in Blade

```php
use Vote4Rewards\Pingback\LinkGenerator;

// In your controller
public function showVotePage()
{
    $linkGenerator = new LinkGenerator();
    
    // Get your server UUID from config or database
    $serverUuid = config('vote4rewards.server_uuid'); // e.g., '550e8400-e29b-41d4-a716-446655440000'
    
    return view('vote', [
        'voteUrl' => $linkGenerator->generateVoteLink(
            $serverUuid,
            auth()->user()->username
        )
    ]);
}
```

## Why UUID Instead of Slug?

Using UUIDs instead of server slugs makes integration much simpler:

- **No need to know category**: The UUID automatically redirects to the correct server page
- **Stable identifier**: Your UUID never changes, even if you rename your server
- **Simple integration**: Just one parameter to configure instead of tracking category + slug

## Testing Your Integration

vote4rewards.de provides a built-in testing tool in your server dashboard:

1. Navigate to your server in the dashboard
2. Click the **"Package"** button
3. Click **"Test Pingback"** in the top right
4. The system will send a test callback to your configured URL

**Test Data Sent:**

- `voter_id`: TestPlayer123 (Unique identifier of the voter)
- `server_uuid`: Your server uuid
- `ip_address`: IP address of the voter
- `voted_at`: 2024-06-01T12:00:00Z
- `test`: true (indicates this is a test callback)

This helps you verify your integration works before going live!

## Security & Verification

### Webhook Signature Verification (HMAC-SHA256)

**Every callback is signed with an HMAC-SHA256 signature.** This ensures:
- ✅ Only vote4rewards.de can send callbacks to your server
- ✅ Callbacks cannot be intercepted or modified
- ✅ You can reject unauthorized requests (e.g., from Insomnia, Postman, attackers)

The signature is sent in the `X-Webhook-Signature` header:

```
X-Webhook-Signature: sha256=abc123def456...
```

**Your webhook secret is unique per server** and is automatically generated when you create your server on vote4rewards.de. Find it in your server dashboard under **"Package"** → **"Integration"**.

#### Verification Example

```php
// This is automatically handled by CallbackHandler->handle() method
// But here's how it works internally:

$payloadJson = json_encode($data);
$webhookSecret = 'your_secret_from_dashboard';
$expectedSignature = hash_hmac('sha256', $payloadJson, $webhookSecret);

$signature = $_SERVER['HTTP_X_WEBHOOK_SIGNATURE']; // "sha256=abc123..."
$providedHash = explode('=', $signature)[1]; // Extract hash part

// Use constant-time comparison to prevent timing attacks
$isValid = hash_equals($expectedSignature, $providedHash);
```

#### Obtaining Your Webhook Secret

1. Go to vote4rewards.de dashboard
2. Select the „servers“ on the left
3. Click **„Package“** button on the server
4. Navigate to **„Secret“** tab
5. Copy your **Webhook Secret**

**For security:**
- Store it in `config/vote4rewards.php` or as environment variable: `VOTE4REWARDS_WEBHOOK_SECRET`
- Never commit it to version control
- Regenerate it if compromised

### IP Whitelisting

```php
use Vote4Rewards\Pingback\CallbackVerifier;

$verifier = new CallbackVerifier();

$handler->onVerify(function(array $data, string $ip) use ($verifier) {
    return $verifier->verify($data, $ip);
});
```

## Configuration for vote4rewards.de

When setting up your server on vote4rewards.de, configure your callback URL in the server settings: (the callback route is up to you.)

```bash
https://yourwebsite.com/vote/callback
```

Make sure:

- The URL is publicly accessible
- It accepts POST requests
- HTTPS is enabled (recommended)
- CSRF protection is disabled for this route

## License

Proprietary - For use with vote4rewards.de servers only.
